package gov.va.genisis2.service.impl;

import gov.va.genisis2.common.enums.RoleTypeEnum;
import gov.va.genisis2.common.enums.UserTypeEnum;
import gov.va.genisis2.converter.UserConverter;
import gov.va.genisis2.converter.UserRoleTypeConverter;
import gov.va.genisis2.dao.IRefreshHistoryDao;
import gov.va.genisis2.dao.IRoleTypeDao;
import gov.va.genisis2.dao.IUserLdapDao;
import gov.va.genisis2.dao.IUserManagementDao;
import gov.va.genisis2.dao.IUserRoleTypeDao;
import gov.va.genisis2.dao.IUserTypeDao;
import gov.va.genisis2.exceptions.ErrorEnum;
import gov.va.genisis2.exceptions.GenisisDAOException;
import gov.va.genisis2.exceptions.GenisisServiceException;
import gov.va.genisis2.model.RefreshHistory;
import gov.va.genisis2.model.RoleType;
import gov.va.genisis2.model.User;
import gov.va.genisis2.model.UserCount;
import gov.va.genisis2.model.UserRoleType;
import gov.va.genisis2.model.UserType;
import gov.va.genisis2.service.IUserManagementService;
import gov.va.genisis2.util.FortifyHelper;
import gov.va.genisis2.vo.LdapUser;
import gov.va.genisis2.vo.LdapUserGroup;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

/**
 * The Class UserManagementService.
 *
 * @author Getaneh Kassahun
 */
@Service
public class UserManagementService implements IUserManagementService {

	private static final Logger LOGGER = LoggerFactory.getLogger(UserManagementService.class);
	private static final Logger LOGGERlDAPREFRESH = LoggerFactory.getLogger("gov.va.genisis2.schedule");
	
	private static final String LDAP_ROLE_GROUP_DELIMETER = ", ";

	@Autowired
	private IUserManagementDao userManagementDao;

	@Autowired
	private IRefreshHistoryDao refreshHistoryDao;

	@Autowired
	private IUserLdapDao userLdapDao;

	@Autowired
	private UserConverter userConverter;

	@Autowired
	private UserRoleTypeConverter userRoleTypeConverter;

	@Autowired
	private IUserRoleTypeDao userRoleTypeDao;

	@Autowired
	private IUserTypeDao userTypeDao;

	@Autowired
	private IRoleTypeDao roleTypeDao;

	/**
	 * This method is used to getUserDetailsByEmail.
	 * 
	 * @param email
	 *            The email.
	 * @return User This returns User.
	 */
	@Override
	public User getUserDetailsByEmail(String email) throws GenisisServiceException {
		try {
			return userManagementDao.getUserDetailsByEmail(email);
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying getUserDetailsByEmail.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * This method is used to getUserDetailsById.
	 * 
	 * @param id
	 *            The id.
	 * @return User This returns user.
	 */
	@Override
	public User getUserDetailsById(int id) throws GenisisServiceException {
		try {
			return userManagementDao.getUserDetailsById(id);
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying getUserDetailsById.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * This method is used to getUserRole.
	 * 
	 * @param uid
	 *            The uid.
	 * @return String This returns userRole.
	 */
	@Override
	public String getUserRole(int uid) throws GenisisServiceException {
		try {
			return userManagementDao.getUserRole(uid);
		} catch (Exception ex) {
			LOGGER.error("Exception occured while querying getUserRole.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * This method is used to get user details by user name
	 * 
	 * @param username
	 *            The user name.
	 * @return User This returns user.
	 */
	@Override
	public User getUserDetailsByUsername(String username) throws GenisisServiceException {
		try {
			return userManagementDao.getUserDetailsByUsername(username);
		} catch (Exception ex) {
			LOGGER.error("Exception occured while querying getUserDetailsByUsername.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * Clear all entries from the cache with roleTypeId.
	 * 
	 * @param roleTypeId
	 *            The role Type Id.
	 */
	@CacheEvict(value = "getUserDetailsByRoleTypeId", key = "#roleTypeId")
	public void resetAllEntries(int roleTypeId) {
		// Intentionally blank
	}

	/**
	 * This method is used to get user details by role Type Id
	 * 
	 * @param roleTypeId
	 *            The role Type Id.
	 * @return User This returns list of user.
	 */
	@Override
	@Cacheable(value = "getUserDetailsByRoleTypeId", key = "#roleTypeId")
	public List<User> getUserDetailsByRoleTypeId(int roleTypeId) throws GenisisServiceException {
		try {
			return userManagementDao.getUserDetailsByRoleTypeId(roleTypeId);
		} catch (Exception ex) {
			LOGGER.error("Exception occured while querying getUserDetailsByRoleId.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * This method is used to get count of users on a role
	 * 
	 * @return User This returns count of users on a role.
	 */
	@Override
	public List<UserCount> getUserCountOnRole() throws GenisisServiceException {
		try {
			return userManagementDao.getUserCountOnRole();
		} catch (Exception ex) {
			LOGGER.error("Exception occured while querying getUserCountOnRole.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * This method is used to get latest LDAP user refresh date/time
	 * 
	 * @return User This returns latest LDAP user refresh date/time
	 */
	@Override
	public Date getLatestUserRefreshHistory() throws GenisisServiceException {
		try {
			return refreshHistoryDao.getLatestUserRefreshHistory();
		} catch (Exception ex) {
			LOGGER.error("Exception occured while querying getLatestUserRefreshHistory.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	@Override
	public int createUserRefreshHistory(RefreshHistory refreshHistory) throws GenisisServiceException {
		try {
			return refreshHistoryDao.createUserRefreshHistory(refreshHistory);
		} catch (Exception ex) {
			LOGGER.error("Exception occured while querying createUserRefreshHistory.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	@Override
	public RefreshHistory updateUserRefreshHistory(RefreshHistory refreshHistory) throws GenisisServiceException {
		try {
			return refreshHistoryDao.updateUserRefreshHistory(refreshHistory);
		} catch (Exception ex) {
			LOGGER.error("Exception occured while querying updateUserRefreshHistory.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	@Override
	public void refreshUsers(String username) throws GenisisServiceException {
		List<LdapUserGroup> ldapUserGroups;
		List<User> users;
		List<String> updatedUserNames = new ArrayList<>();

		try {
			ldapUserGroups = userLdapDao.getAllUsersByGroup();
			users = userManagementDao.getAllUsers();

			// ignore users with multiple roles
			Set<String> duplicateUsers = this.ignoreUsersWithMultipleRoles(ldapUserGroups, users);

			// create a user map
			Map<String, User> userMap = createUserMap(users);

			// add/update user details in USERS table with LDAP
			for (LdapUserGroup ldapUserGroup : ldapUserGroups) {
				if (null != ldapUserGroup) {
					compareUsers(ldapUserGroup, userMap, updatedUserNames);
				}
			}

			if (!duplicateUsers.isEmpty()) {
				updatedUserNames.addAll(duplicateUsers);
			}
			// update user status to 'DISABLE'
			updateUserStatus(updatedUserNames);
		} catch (Exception ex) {
			LOGGER.error("Exception occured while querying getAllUsers from LDAP", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	private void compareUsers(LdapUserGroup ldapUserGroup, Map<String, User> userMap, List<String> updatedUserNames) throws GenisisDAOException {
		String userGroupName = ldapUserGroup.getGroupName();
		List<LdapUser> ldapUsers = ldapUserGroup.getUsers();
		User user;
		User persistentUser;

		if (null == ldapUsers || ldapUsers.isEmpty()) {
			return;
		}

		for (LdapUser ldapUser : ldapUsers) {
			if (null == ldapUser) {
				LOGGER.error("ldapUser object is null");
				continue;
			}

			// no users in DB, every LDAP user is a new user
			if (null == userMap || userMap.isEmpty()) {
				user = userConverter.convert(ldapUser);
			} else {
				user = userMap.get(ldapUser.getUsername());

				// new user
				if (null == user) {
					user = userConverter.convert(ldapUser);
				} else {
					// existing user
					userConverter.populate(user, ldapUser);
				}
			}

			// persist user
			persistentUser = userManagementDao.saveOrUpdate(user);

			// add user Type and role Type
			UserRoleType userRoleType = userRoleTypeDao.getUserRoleTypeByUserId(persistentUser.getUserId());
			UserType userType = userTypeDao.getUserType(UserTypeEnum.EMPLOYEE.getId());
			RoleType roleType = roleTypeDao.getRoleType(RoleTypeEnum.getRoleTypeEnum(userGroupName));

			userRoleType = userRoleTypeConverter.populate(userRoleType, persistentUser, userType, roleType);
			userRoleTypeDao.saveOrUpdateUserRoleType(userRoleType);

			// add ldap username to list updatedUserNames
			updatedUserNames.add(persistentUser.getUsername());
		}
	}

	private Map<String, User> createUserMap(List<User> users) {
		Map<String, User> userMap;
		if (null == users || users.isEmpty()) {
			return null;
		} else {
			userMap = new HashMap<>();
			for (User user : users) {
				String userName = user.getUsername();
				String upperCaseUserName = (null == userName || userName.isEmpty()) ? StringUtils.EMPTY : userName.trim().toUpperCase();
				user.setUsername(upperCaseUserName);

				String emailId = user.getEmailId();
				user.setEmailId((null == emailId || emailId.isEmpty()) ? StringUtils.EMPTY : emailId.trim());

				userMap.put(upperCaseUserName, user);
			}
		}

		return userMap;
	}

	private void updateUserStatus(List<String> updatedUserNames) throws GenisisServiceException {
		boolean flag = null == updatedUserNames || updatedUserNames.isEmpty();
		if (flag) {
			return;
		}

		try {
			userManagementDao.updateUserStatus(updatedUserNames);
		} catch (Exception ex) {
			LOGGER.error(ErrorEnum.USER_DAO_EXP_UPDATESTATUS.getErrorMessage(), ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * To ignore Users With Multiple Roles
	 * 
	 * @param ldapUserGroups
	 * @param users
	 * 
	 * 
	 */
	private Set<String> ignoreUsersWithMultipleRoles(List<LdapUserGroup> ldapUserGroups, List<User> users) {
		List<String> allUserList = new ArrayList<>();

		// Get a List of Users from all groups
		for (LdapUserGroup ldapUser : ldapUserGroups) {
			for (LdapUser ldapUsers : ldapUser.getUsers()) {
				if (ldapUsers != null && !StringUtils.isBlank(ldapUsers.getUsername()))
					allUserList.add(ldapUsers.getUsername());
			}
		}

		Set<String> duplicateUsers = this.findDuplicates(allUserList);

		if (null != duplicateUsers || !duplicateUsers.isEmpty()) {
			removeUserFromLdapUserGroupAndUsers(duplicateUsers, ldapUserGroups, users);
		}
		return duplicateUsers;
	}

	/**
	 * To remove User From Ldap User Group And Users
	 * 
	 * @param duplicateUsers
	 * @param ldapUserGroups
	 * @param users
	 * 
	 * 
	 */
	private void removeUserFromLdapUserGroupAndUsers(Set<String> duplicateUsers, List<LdapUserGroup> ldapUserGroups, List<User> users) {
		Set<String> logOnlyOnce = new HashSet<>();
		Map<String, String> ignoredUsers = new HashMap();
		if (null != duplicateUsers) {
			for (LdapUserGroup ldapGroupUser : ldapUserGroups) {
				// Remove User with multiple Role from LdapUserGroup object
				removeUserFromLdapUserGroup(ldapGroupUser, duplicateUsers, logOnlyOnce, ignoredUsers);
				
				// Remove User with multiple Role from users object
				removeUserfromUsers(duplicateUsers, users);
			}
		}
		// Log users could not be added to the database
		if (ignoredUsers != null && ignoredUsers.size() > 0) {
			Iterator it = ignoredUsers.entrySet().iterator();
			while (it.hasNext()) {
				Map.Entry pair = (Map.Entry) it.next();
				LOGGERlDAPREFRESH.info("The user " + FortifyHelper.getCleanStringInput((String)pair.getKey()) + " has multiple roles defined in LDAP server. This user will not be added to the database. Roles: " + FortifyHelper.getCleanStringInput((String)pair.getValue()));
			}
		}
	}

	/**
	 * To remove User From Ldap User Group
	 * 
	 * @param ldapGroupUser
	 * @param duplicateUsers
	 * 
	 * 
	 */
	private void removeUserFromLdapUserGroup(LdapUserGroup ldapGroupUser, Set<String> duplicateUsers, Set<String> logOnlyOnce, Map<String, String> ignoredUsers) {
		if (ldapGroupUser != null) {
			
			// Remove User with multiple Role from LdapUserGroup
			for (Iterator<LdapUser> iterator = (ldapGroupUser.getUsers()).iterator(); iterator.hasNext();) {
				LdapUser ldapUser = iterator.next();
				if (ldapUser != null && !duplicateUsers.toString().isEmpty() && StringUtils.contains(duplicateUsers.toString(), ldapUser.getUsername())) {
				String 	mapValue = ignoredUsers.get(ldapUser.getUsername());
					if (StringUtils.isBlank(mapValue)) {
						ignoredUsers.put(ldapUser.getUsername(), ldapGroupUser.getGroupName());
					} else {
						ignoredUsers.put(ldapUser.getUsername(), mapValue + LDAP_ROLE_GROUP_DELIMETER + ldapGroupUser.getGroupName());
					}
					iterator.remove();
				}
			}
		}
	}


	/**
	 * To remove User from Users
	 * 
	 * @param duplicateUsers
	 * @param users
	 * 
	 * 
	 */
	private void removeUserfromUsers(Set<String> duplicateUsers, List<User> users) {
		// Remove User with multiple Role from users object
		for (Iterator<User> iterator = users.iterator(); iterator.hasNext();) {
			User ldapUser = iterator.next();
			if (ldapUser != null && !duplicateUsers.toString().isEmpty() && StringUtils.contains(duplicateUsers.toString(), ldapUser.getUsername())) {
				// iterator.remove();
			}
		}
	}

	/**
	 * To find Duplicate user
	 * 
	 * @param groupsuserList
	 * @return string
	 * 
	 */
	private Set<String> findDuplicates(List<String> groupsuserList) {
		final Set<String> duplicateusers = new HashSet<>();

		final Set<String> set1 = new HashSet<>();
		for (String userName : groupsuserList) {
			if (!set1.add(userName)) {
				duplicateusers.add(userName);
			}
		}
		return duplicateusers;
	}
}
